#!/usr/local/bin/perl -wT
# Copyright Andru Luvisi, 1999
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program (see the file COPYING); if not, write to the
# Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA

$| = 1;

open(STDERR, ">&STDOUT") || die "Can't redirect stderr";

use CGI;
use CGI::Carp qw(fatalsToBrowser);
use POSIX qw(:fcntl_h);
use English;
use strict;

use vars qw($query $fcnfile $query_dir $fcntext %handlers
            $response $errortext $location $delim $tag);
use vars qw($authorized_uid);


# CONFIGURATION SECTION
$authorized_uid = 1031;

if(!$EFFECTIVE_USER_ID && $REAL_USER_ID != $authorized_uid) {
  die("Unauthorized user $REAL_USER_ID");
}


# this handles all form data for us...
$query = CGI->new;

# untaint $fcnfile, since it's given to us by the web server.
$ENV{"PATH_TRANSLATED"} =~ m/(.*)/;
$fcnfile = $1;

# if we are running as root, change both uid and euid to be that of
# the owner of the fcnfile.
if(!$EFFECTIVE_USER_ID) {
 # stat the file...
 my($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
                      $atime,$mtime,$ctime,$blksize,$blocks)
                          = stat($fcnfile);

 if(!defined($uid)) {
  print "Content-type: text/html\r\n\r\n";
  print "<h1>Error stating fcnfile</h1>\n";
  exit(1);
 }

 $REAL_USER_ID      = $uid;
 $EFFECTIVE_USER_ID = $uid;
}

# get the name of the directory the fcnfile is in.
# this is used in savecustom and savedata.
$query_dir = $fcnfile;
$query_dir =~ s/[^\/]+$//;

# this is to untaint $ENV{"PATH"} so that exec below won't complain.
$ENV{"PATH"} = "/bin:/usr/bin:/usr/local/bin";

# slirp in the whole fcnfile into the string $fcntext.
open(FCNFILEHANDLE, "$fcnfile") || die("Can't open $fcnfile");
undef $INPUT_RECORD_SEPARATOR;
$fcntext = <FCNFILEHANDLE>;
$fcntext =~ s/\r\n/\n/g; # handle DOS text
$fcntext =~ s/\r/\n/g;   # handle Mac text
close(FCNFILEHANDLE);

# used for looking up the handler for each tag.

%handlers = (
 "mail"       => \&mail,
 "response"   => \&response,
 "require"    => \&require,
 "error"      => \&error,
 "savecustom" => \&savecustom,
 "savedata"   => \&savedata,
 "redirectto" => \&redirectto,
 "replace"    => \&replace,
);

# initialize variables...
# this is appended to by the <response> container.  We send it
# back to the user if everything works and there are no redirections.
$response = "";   

# this is set by the <error> container.  We send it back to
# the use of a <require ...> tag fails.
$errortext = "";  

# if this is set when we finish, and everything works, we redirect
# the user to this location.  it is set by the <redirectto ...> tag.
$location = "";

# this is used to delimit substitutions in the body of various containers.
# it can be changed with the <replace> container.
$delim = "%";

# this is the main loop.  find a tag, and call the appropriate handler
# if there is one.
while( $fcntext =~ m/<(\w+)(.*?)>/gsci ) {
 $tag = lc($1);
 $handlers{$tag} && &{$handlers{$tag}}($tag, $2);
}

# finish up... if we are to redirect the user, send a Location header,
# otherwise, send the response as text/html.
if($location) {
 print("Location: $location\r\n\r\n");
} else {
 print("Content-type: text/html\r\n\r\n");
 print $response, "\n";
}

exit(0);


# GENERIC SUBROUTINES

# string getbody( string $tagname );
# this is called to retrieve the body of a container by the handler
# for that container.  $tagname is the name of the container, and is
# used to find the closing tag.
sub getbody {
 my($tagname) = shift;
 if( $fcntext =~ m/\G(.*?)(<\/$tagname>)/gsci ) {
  return $1;
 }

 return "";

}

# string substitute( string $text );
# this is used to replace all occurences of "%var%" in $text with
# the value of the submitted form variable "var".
sub substitute {
 my($text) = $_[0];

 $text =~ s/$delim(\w+)$delim/ $query->param($1) /ge;
 return $text;

}

# typglob open_output( string $file [ , string $mode ] );
# this is used by savecustom and savedata to open an output file.
# $mode may be blank, "append", "new", or "unique".  if $mode is 
# blank or "append", $file is opened in append mode.  if $mode is
# "new", $file is truncated and opened in write mode.  if $mode is
# "unique", $file is used as the base name for a new file which doesn't
# yet exist.  the return value is a typeglob containing the filehandle
# which was opened.
sub open_output {
 my($file) = $query_dir . $_[0];
 my($opened, $num) = (0, time() . $PROCESS_ID);
 local($_) = $_[1];
 local(*FILE);

 # truncate the output file if $mode contains "new".
 if(/new/i) {
  $file = "> " . $file;
  open(FILE, $file) || return undef;
  return *FILE;
 }

 # this is the tricky one.  since we are in a multi-programmed environment,
 # we can not check for the existance of a file and create it atomically.
 # we must rely on the operating system to do this for us.  the O_CREAT
 # flag tells the system to create the file if it does not exist, and the
 # O_EXCL flag tells the system to fail to open the file if it already
 # exists.  since $num is already almost guaranteed to be unique, we will
 # "spin" in this loop very rarely and very briefly.
 if(/unique/i) {
  while(!$opened) {
   if(sysopen(FILE, "$file.$num", O_WRONLY | O_CREAT | O_EXCL, 0666)) {
    $opened = 1;
   } else {
    $num++;
    next;
   }
  }
  return *FILE;
 }
  
 # append if $mode contains "append" or is blank.
 if(/append/i || /^\W*$/) {
  $file = ">> " . $file;
  open(FILE, $file) || return undef;
  return *FILE;
 }

 die("Invalid argument to SAVECUSTOM or SAVEDATA");

}

# TAG AND CONTAINER HANDLERS.

# <mail> container.  treat contents of the container as the full
# message, including headers.  If there is no Return-Path, set
# the Return-Path header from the From header, just to be sure it
# will work with things like listbots.
sub mail {
 # get the body of the <mail> tag into $tagbody.
 my($tagbody) = getbody("mail");

 # split $tagbody into a header and a body.
 my($mailhead, $mailbody) = ($tagbody =~ m/(.*?\n)\n(.*)/gs);
 my(%headers);
 my($name, $child_id);

 # split up the headers and store them into the headers hash.
 %headers = ($mailhead =~ m/^(\S+): (.*\n(?:[ \t].*\n)*)/mg);
 
 # fork the child.
 if(!defined($child_id = open(MAIL, "|-"))) {
  die("Can't fork mail process!");
 }

 # the child runs sendmail.  the -t is so that it will take its
 # destination addresses (To, Cc, Bcc) from the headers we feed it,
 # thereby saving us from having to parse the headers ourselves.
 if($child_id == 0) {
  # without the if, perl warns that the exit is unlikely to be reached.
  if(1) { exec("/usr/lib/sendmail -t"); }
  exit(1);
 }

 # we have to do the substitutions one header at a time so that form
 # values containing multiple lines don't create new headers.
 foreach $name (keys %headers) {
  $headers{$name} =~ s/\n[ \t]/ /g;
  $headers{$name} = substitute($headers{$name});
  $headers{$name} =~ s/\n(?=.)/ /g;
  $headers{$name} =~ s/\n$/\r\n/g;
  print MAIL "$name: $headers{$name}"; # CRLF is already included.
 }

 # if no Return-Path was specified, set it from the From header, so
 # that the envelope return address will also be the one specified
 # in the fcnfile.
 if(!$headers{"Return-Path"}) {
  print MAIL "Return-Path: $headers{'From'}";
 }

 # print out the body of the message.  a single \r\n serves to terminate
 # the headers since they each have their own \r\n at the end.
 $mailbody =~ s/\n/\r\n/g;
 print MAIL "\r\n", substitute($mailbody);
 close(MAIL);
 
}

# handle the <response> container.  get the body, and do the substitutions.
sub response {
 $response .= substitute(getbody("response"));
}

# <require ...>
# see if the user entered all of the form values which are required.
# if any are missing, print out the current value of $errortext 
# (set with the <error> container) and don't process any more of the
# fcnfile.
sub require {
 while($_[1] =~ m/(\w+)/g) {
  if( ! $query->param($1) ) {
   print "Content-type: text/html\r\n\r\n";
   print "$errortext";
   exit(0);
  }
 }
}

# <error>
# assign (not append) the new value of $errortext.
sub error {
 $errortext = substitute(getbody("error"));
}

# <savecustom "<filename>" [ APPEND | NEW | UNIQUE ]>...</savecustom>
# the body gets substituted in the same way as that of <response>
# and <mail>.
sub savecustom {
 my($file, $junk, $arg) = ($_[1] =~ m/"([^"]+)"(\s+(\w+))?/g);
 my($text) = substitute(getbody("savecustom"));

 local(*OUTPUT_FILE) = open_output($file, $arg);
 print OUTPUT_FILE $text;
 close(OUTPUT_FILE);
}

# <savedata "<filename>" [ APPEND | NEW | UNIQUE ]>...</savecustom>
# the output is tab delimited, and the body of this tag is just the
# bare names of form fields to save.
sub savedata {
 my($file, $junk, $arg) = ($_[1] =~ m/"([^"]+)"(\s+(\w+))?/g);
 my($text) = getbody("savedata");
 my(@fields) = split(/\s+/, $text);

 local(*OUTPUT_FILE) = open_output($file, $arg);
 print OUTPUT_FILE join("\t", map { $query->param($_); } @fields), "\n";
 close(OUTPUT_FILE);
}

# <redirectto href="http://www.example.com/">
# set $location so that if we finish executing properly, the user
# will be redirected to http://www.example.com/ .
sub redirectto {
 ($location) = ($_[1] =~ m/href="([^"]+)"/ig);
}

# <replace>...</replace>
# change the delimiter to mark substitutions.
sub replace {
 my($body) = getbody("replace");
 ($body) =~ m/(\S+)/g;
 $delim = $body if $body;
}

